<?php
/**
 * @package     Joomla.Plugin
 * @subpackage  e4j.VikUpdater
 * @copyright   Copyright (C) 2023 e4j - Extensionsforjoomla.com. All Rights Reserved.
 * @license     GNU General Public License version 2 or later; see LICENSE.txt
 */

// No direct access to this file
defined('_JEXEC') or die('Restricted access');

include dirname(__FILE__) . DIRECTORY_SEPARATOR . 'libraries' . DIRECTORY_SEPARATOR . 'autoload.php';

use E4J\VikUpdater\Software\Version as SoftwareVersion;

/**
 * VikUpdater CMS plugin.
 *
 * @since 1.0
 */
class plgE4jVikupdater extends JPlugin
{
	/**
	 * Load the language file on instantiation. Note this is only available in Joomla 3.1 and higher.
	 * If you want to support 3.0 series you must override the constructor
	 *
	 * @var    bool
	 * @since  3.1
	 */
	protected $autoloadLanguage = true;

	/**
	 * The application instance.
	 * 
	 * @var    JApplication
	 * @since  1.5
	 */
	protected $app;

	/**
	 * The database driver instance.
	 * 
	 * @var    JDatabaseDriver
	 * @since  1.5
	 */
	protected $db;

	/**
	 * Method used to check whether there is a plugin able
	 * to support external updates.
	 * 
	 * @return 	bool  Return always true.
	 *
	 * @since 	1.2
	 */
	public function onUpdaterSupported()
	{
		return true;
	}

	/**
	 * Return the stored version contents of the program.
	 *
	 * @param 	mixed   $args  The array|object containing the parameters to check.
	 *
	 * @return 	object  The contents of the version stored.
	 *
	 * @throws 	RuntimeException
	 *
	 * @since 	1.2
	 */
	public function onGetVersionContents($args)
	{
		// may raise an exception in case the software is not valid
		$software = $this->getSoftware($args);

		if ($software === null)
		{
			return null;
		}

		return $software->content;
	}

	/**
	 * Check the current version of the program which is triggering the plugin.
	 *
	 * @param 	mixed   $args  The array|object containing the parameters to check.
	 *
	 * @return 	object 	The response about the version check.
	 *
	 * @throws 	RuntimeException
	 *
	 * @since 	1.2
	 */
	 public function onCheckVersion($args)
	 {
		$software = E4J\VikUpdater\Software\Factory::create($args);

		if ($software->check() === false)
		{
			throw new RuntimeException('Illegal arguments.');
		}

		// get connection handler (may raise an exception in case there is no driver)
		$conn = new E4J\VikUpdater\Network\Http();

		// call the end-point and get a response
		$response = $conn->get('&sku=' . $software->getSku());

		// create a response object to return
		$obj = new stdClass;
		$obj->status = 1;

		if ($response->code !== 200)
		{
			$obj->status = 0;
			$obj->error  = $response->body;
		}
		else
		{
			// parse plugin XML manifest file
			$xml = @simplexml_load_string($response->body);

			if (!$xml)
			{
				$obj->status = 0;
				$obj->error  = 'Cannot parse manifest XML file.';
			}
			else
			{
				$obj->response = new stdClass;
				$obj->response->status    = 1;
				$obj->response->changelog = [];

				// register download URL
				$obj->response->downloadurl = (string) $xml->update->downloads->downloadurl;

				// compare the latest version with the current one
				$obj->response->compare = version_compare((string) $xml->update->version, $software->getVersion());

				if ($obj->response->compare > 0)
				{
					$obj->response->title      = JText::_('PLG_VIKUPDATER_NEW_VERSION_AVAILABLE_TEXT');
					$obj->response->shortTitle = JText::_('PLG_VIKUPDATER_NEW_VERSION_AVAILABLE_SHORT');

					// build changelog by displaying the link to reach the official blog page
					$obj->response->changelog = $this->buildChangelog($xml);
				}
				else if ($obj->response->compare == 0)
				{
					$obj->response->title      = JText::_('PLG_VIKUPDATER_UP_TO_DATE_TEXT');
					$obj->response->shortTitle = JText::_('PLG_VIKUPDATER_UP_TO_DATE_SHORT');
				}
				else
				{
					$obj->response->title      = JText::_('PLG_VIKUPDATER_INVALID_VERSION_TEXT');
					$obj->response->shortTitle = JText::_('PLG_VIKUPDATER_INVALID_VERSION_SHORT');
				}
			}
		}

		// store the version
		$this->storeVersion($software, $obj);

		return $obj;
	}

	/**
	 * Launch the update process.
	 *
	 * @param 	mixed   $args  The array|object containing the parameters to check.
	 *
	 * @return 	object  The response about the update.
	 *
	 * @throws 	RuntimeException
	 *
	 * @since 	1.2
	 */
	public function onDoUpdate($args)
	{
		$software = E4J\VikUpdater\Software\Factory::create($args);
		
		if ($software->check() === false)
		{
			throw new RuntimeException('Illegal arguments.');
		}

		// get software arguments
		$args = $this->getSoftware($software);

		if (!$args || empty($args->content->response->downloadurl))
		{
			throw new RuntimeException('Unable to find the URL to download the update.');
		}

		// get Joomla version
		$jv = new JVersion;
		
		if (version_compare($jv->getShortVersion(), '4.0') >= 0)
		{
			// load installer model (J4)
			$model = JModelLegacy::getInstance('install', 'installer');
		}
		else
		{
			// load installer model (J3)
			JModelLegacy::addIncludePath(JPATH_ADMINISTRATOR . DIRECTORY_SEPARATOR . 'components' . DIRECTORY_SEPARATOR . 'com_installer' . DIRECTORY_SEPARATOR . 'models');
			$model = JModelLegacy::getInstance('install', 'InstallerModel');	
		}

		// throw new exception
		if (!$model)
		{
			// installer model not found
			throw new RuntimeException('Unable to load the installer model.');
		}

		// build download URI
		$downloadUri = new JUri($args->content->response->downloadurl);

		// include arguments for the update validation
		$downloadUri->setVar('domain', base64_encode(E4J\VikUpdater\Network\Server::base()));
		$downloadUri->setVar('ip', E4J\VikUpdater\Network\Server::get('REMOTE_ADDR'));

		// inject installation roles within the request so that the model will be
		// able to properly use them, since the method doesn't accept any arguments
		$this->app->input->set('installtype', 'url');
		$this->app->input->set('install_url', (string) $downloadUri);

		// auto-load com_installer administrator language
		$lang = JFactory::getLanguage();
		$lang->load('com_installer', JPATH_ADMINISTRATOR, $lang->getTag(), true);

		// Trigger the installation method. In case of error, the process will be 
		// aborted by the registered "onInstallerAfterInstaller" event (J4 only).
		if (!$model->install())
		{
			// an error occurred probably before starting the update process...
			$this->abortUpdateProcess();
		}

		// wipe out the user state before the end of the process, otherwise if you
		// try to visit the installer page a message would be displayed
		$this->app->setUserState('com_installer.redirect_url', '');
		$this->app->setUserState('com_installer.message', '');
		$this->app->setUserState('com_installer.extension_message', '');

		// remove version on success
		$this->removeVersion($software);

		return true;
	}

	/**
	 * Triggers at the end of the installation process.
	 * 
	 * @param 	mixed   $model      The installer model.
	 * @param 	mixed   &$package   The update package.
	 * @param 	mixed   $installer  The installer helper.
	 * @param 	bool    &$result    True in case of success, false otherwise.
	 * @param 	string  &$msg       The fetched error/success message.
	 * 
	 * @return  void
	 * 
	 * @throws 	RuntimeException
	 * 
	 * @since 	1.3
	 */
	public function onInstallerAfterInstaller($model, &$package, $installer, &$result, &$msg)
	{
		if (!$result)
		{
			// an error occurred, abort update process
			$this->abortUpdateProcess($msg);
		}
	}

	////////////////////////////
	///// HELPER FUNCTIONS /////
	////////////////////////////

	/**
	 * Return the details of the stored software.
	 * The details stored are ignored if the last update is too old.
	 *
	 * @param 	mixed   $software  The array|object containing the parameters to check.
	 *
	 * @return 	object  The details about the software stored.
	 *
	 * @throws 	RuntimeException
	 */
	protected function getSoftware($software)
	{
		if (!$software instanceof SoftwareVersion)
		{
			$software = E4J\VikUpdater\Software\Factory::create($software);
		}

		if ($software->check() === false)
		{
			throw new RuntimeException('Illegal arguments.');
		}

		$days = $this->params->get('check_days', 7);

		$q = $this->db->getQuery(true);

		$q->select('*')
			->from($this->db->qn('#__vikupdater_plugin_software'))
			->where($this->db->qn('digest') . ' = ' . $this->db->q($software->digest()))
			->where($this->db->qn('last_update') . ' > ' . (time() - $days * 86400));

		$this->db->setQuery($q, 0, 1);
		$obj = $this->db->loadObject();

		if ($obj)
		{
			$obj->content = json_decode($obj->content);
		}

		return $obj;
	}

	/**
	 * Register the version of the program to recover it.
	 *
	 * @param   SoftwareVersion  $software  The software version.
	 * @param   mixed            $content   The contents to store.
	 *
	 * @return  bool  True on success, otherwise false.
	 *
	 * @throws 	RuntimeException
	 */
	protected function storeVersion(SoftwareVersion $software, $content)
	{
		// may raise an exception in case the software is not valid
		$obj = $this->getSoftware($software);

		$q = $this->db->getQuery(true);

		$success = false;

		if ($obj === null)
		{
			// insert

			$q->insert($this->db->qn('#__vikupdater_plugin_software'))
				->columns([
					$this->db->qn('digest'),
					$this->db->qn('content'),
					$this->db->qn('last_update'),
				])
				->values(
					$this->db->q($software->digest()) . ',' .
					$this->db->q(json_encode($content)) . ',' .
					time()
				);

			$this->db->setQuery($q);
			$this->db->execute();

			$success = (bool) $this->db->insertid();
		}
		else
		{
			// update

			$q->update($this->db->qn('#__vikupdater_plugin_software'))
				->set($this->db->qn('content') . ' = ' . $this->db->q(json_encode($content)))
				->set($this->db->qn('last_update') . ' = ' . time())
				->where($this->db->qn('id') . ' = ' . $obj->id);

			$this->db->setQuery($q);
			$this->db->execute();

			$success = (bool) $this->db->getAffectedRows();
		}

		return $success;
	}

	/**
	 * Remove the version of the program to re-check it.
	 *
	 * @param   SoftwareVersion  $software  The software version.
	 * 
	 * @return  void
	 *
	 * @throws  RuntimeException
	 */
	protected function removeVersion(SoftwareVersion $software)
	{
		// may raise an exception in case the software is not valid
		$obj = $this->getSoftware($software);

		if ($obj !== null)
		{
			$q = $this->db->getQuery(true);

			$q->delete($this->db->qn('#__vikupdater_plugin_software'))
				->where($this->db->qn('id') . ' = ' . $obj->id);

			$this->db->setQuery($q);
			$this->db->execute();
		}
	}

	/**
	 * Helper method used to build a changelog.
	 * 
	 * @param 	XMLDocument  $xml  The update manifest.
	 * 
	 * @return 	array  A list of changelog nodes.
	 * 
	 * @since  1.3
	 */
	protected function buildChangelog($xml)
	{
		$title = new stdClass;
		$title->tag = 'h2';
		$title->content = $xml->update->name . ' ' . $xml->update->version;

		$label = new stdClass;
		$label->tag = 'p';
		$label->content = JText::_('PLG_VIKUPDATER_CHANGELOG_PARAGRAPH');

		$link = new stdClass;
		$link->tag = 'a';
		$link->content = (string) $xml->update->infourl;
		$link->attributes = new stdClass;
		$link->attributes->href = (string) $xml->update->infourl;
		$link->attributes->target = '_blank';

		return [
			$title,
			$label,
			$link,
		];
	}

	/**
	 * Helper method used to abort the update process and to fetch
	 * the error messages that might have been enqueued by Joomla.
	 * 
	 * @param 	string 	$msg  An optional message.
	 * 
	 * @return 	void
	 * 
	 * @throws 	RuntimeException
	 * 
	 * @since 	1.3
	 */
	protected function abortUpdateProcess($msg = '')
	{
		if ($msg)
		{
			// wrap message in a paragraph
			$msg = '<p>' . strip_tags($msg) . '</p>';
		}

		$code = 500;

		// iterate all the error messages
		foreach ($this->app->getMessageQueue($clear = true) as $record)
		{
			$record = (array) $record;

			if (empty($record['message']) || empty($record['type']))
			{
				// ignore in case the message or the type are empty
				continue;
			}

			// accpet only errors and warnings
			if (!in_array($record['type'], ['error', 'warning']))
			{
				continue;
			}

			// include a new message
			$msg .= '<p>' . $record['message'] . '</p>';

			// try to detect whether the error message contains an HTTP code between 400 and 599
			if (preg_match("/\b[45][\d]{2,2}\b/", $record['message'], $match))
			{
				// use the code found
				$code = (int) end($match);
			}
		}

		if ($code === 403 || $code === 401)
		{
			// possibly detected a forbidden error

			if ($msg)
			{
				// include a separator
				$msg .= '<hr />';
			}

			// add a generic message
			$msg .= JText::_('PLG_VIKUPDATER_EXPIRED_LICENSE_ERROR');
		}

		// finalize the update with an error
		throw new RuntimeException($msg, $code);
	}
}
